/*
 * Routines to support Link Aggregation for ethernet
 */

#include <netinet/in.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_lag.h"
#include "lf_topo_map.h"
#include "lf_fma_flags.h"

/*
 * Local routines
 */
static int lf_lag_lookup_lag_id(struct lf_fabric *fp,
    struct lf_lag_id_def **hash_table, unsigned char *lag_id);

/*
 * Assign mag_ids to all NICs that need them
 */
int
lf_lag_assign_mag_ids(
  struct lf_fabric *fp)
{
  struct lf_lag_id_def *lidp;
  struct lf_lag_id_def **lag_hash;
  struct lf_host *hp;
  struct lf_nic *nicp;
  int next_mag_id;
  int lidi;
  int h;
  int n;

  lag_hash = NULL;

  /* clear and restart lag_id table */
  LF_FREE(fp->lag_ids);
  fp->num_lag_ids = 1;
  fp->lag_tbl_size = fp->num_hosts;	/* guess at how many we might need */
  LF_CALLOC(fp->lag_ids, struct lf_lag_id_def, fp->lag_tbl_size);

  /* Allocate hash table for lag_ids */
  LF_CALLOC(lag_hash, struct lf_lag_id_def *, LF_LAG_HASH_TABLE_SIZE);

  /*
   * Loop through all NICs, for each one with a lag_id, check to see if
   * it is one we have seen before.  If not, create a new entry for it,
   * otherwise just increment the reference count.
   * In the end, only LAG_IDs with reference count >1 will receive
   * non-zero MAG_IDs.
   */
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    /* Check the LAG_ID of each NIC */
    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      /* only compute mag_id if host can do it */
      if (hp->fma_flags & FMA_FLAG_CAN_DO_LAG) {

	/* Look up LAG_ID descriptor, creating one if new */
	lidi = lf_lag_lookup_lag_id(fp, lag_hash, nicp->nic_lag_id);
	if (lidi == -1) {
	  LF_ERROR(("Error looking up lag_id"));
	}
	lidp = &(fp->lag_ids[lidi]);

	/* save index of lag_id descriptor */
	nicp->lag_id_index = lidi;

	/* up the reference count */
	++lidp->ld_ref_cnt;

      /* mag_id is zero if host cannot do LAG */
      } else {
	nicp->mag_id = 0;
	nicp->lag_id_index = 0;
      }
    }
  }

  /*
   * All lag_ids are now hashed - step through the hash table, assigning
   * mag_ids to all holders of lag_ids with reference counts > 1
   */
  next_mag_id = 1;
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    /* only compute mag_id if host can do it */
    if (hp->fma_flags & FMA_FLAG_CAN_DO_LAG) {

      /* Check the ref_cnt of each NIC's lag_id */
      for (n=0; n<hp->num_nics; ++n) {
	nicp = hp->nics[n];

	/* get lag_id def struct for this NIC */
	lidp = &(fp->lag_ids[nicp->lag_id_index]);

	/* Assign mag_id to this lag_id if not already assigned */
	if (lidp->ld_ref_cnt > 1) {
	  if (lidp->ld_mag_id == 0) {
	    lidp->ld_mag_id = next_mag_id;
	    ++next_mag_id;
	  }
	  nicp->mag_id = lidp->ld_mag_id;
	} else {
	  nicp->mag_id = 0;
	}
      }
    }
  }

#ifdef FM_TEST_LAG
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      if (hp->fma_flags & FMA_FLAG_CAN_DO_LAG) {
	printf(LF_MAC_FORMAT ", LAG_ID=\"%s\", mag_id=%d\n",
	    LF_MAC_ARGS(nicp->mac_addr),
	    lf_lag_id_string(nicp->nic_lag_id),
	    nicp->mag_id);
      } else {
	printf(LF_MAC_FORMAT ", cannot LAG, mag_id=%d\n",
	    LF_MAC_ARGS(nicp->mac_addr), nicp->mag_id);
      }
    }
  }
#endif

  LF_FREE(lag_hash);
  return 0;

 except:
  LF_FREE(lag_hash);
  return -1;
}

/*
 * Find a lag_id definition in our hash table, creating it if not already
 * existant.
 */
int
lf_lag_lookup_lag_id(
  struct lf_fabric *fp,
  struct lf_lag_id_def **hash_table,
  unsigned char *lag_id)
{
  struct lf_lag_id_def *lidp;
  int lidi;
  int i;

  /* get hash index */
  i = lf_lag_hash(lag_id);

  /* loop through list of lag descriptors with same hash for this lag_id */
  for (lidp = hash_table[i];
       lidp != NULL;
       lidp = lidp->ld_hash_next) {

    /* If we find it, return */
    if (lf_lag_match(lidp->ld_lag_id, lag_id)) {
      return lidp->ld_lag_index;
    }
  }

  /* not found, grab the next one, growing our pool if necessary */
  if (fp->num_lag_ids >= fp->lag_tbl_size) {
    fp->lag_tbl_size += fp->num_hosts;
    fp->lag_ids = (struct lf_lag_id_def *) realloc(fp->lag_ids,
	sizeof(struct lf_lag_id_def) * fp->lag_tbl_size);
    if (fp->lag_ids == NULL) {
      LF_ERROR(("Error growing lag_ids in fabric"));
    }
  }

  /* grab the next descriptor */
  lidi = fp->num_lag_ids;
  lidp = &(fp->lag_ids[lidi]);
  ++fp->num_lag_ids;

  /* fill in the def entry */
  lf_lag_copy(lidp->ld_lag_id, lag_id);
  lidp->ld_mag_id = 0;
  lidp->ld_ref_cnt = 0;
  lidp->ld_lag_index = lidi;

  /* link into table */
  lidp->ld_hash_next = hash_table[i];
  hash_table[i] = lidp;

  return lidi;

 except:
  return -1;
}

/*
 * Generate a string representing a LAG_ID, almost per 802.3 2002 43.3.6.2
 */
char *
lf_lag_id_string(
  unsigned char *lag_id)
{
  static lf_string_t s;
  unsigned short a, b, c, d, e, f;

  /* Copy the potentially misaligned fields. */
  
  memcpy (&a, lag_id+0, 2);
  memcpy (&b, lag_id+8, 2);
  memcpy (&c, lag_id+11, 2);
  memcpy (&d, lag_id+13, 2);
  memcpy (&e, lag_id+21, 2);
  memcpy (&f, lag_id+24, 2);

  /* Build the lag_id string. */

  sprintf(s,
	  "[(%04X," LF_MAC_FORMAT ",%04X,%02X,%04X),"
	  "(%04X," LF_MAC_FORMAT ",%04X,%02X,%04X)]",
	  ntohs(a), LF_MAC_ARGS(lag_id+2), ntohs(b), *(lag_id+10), ntohs(c),
	  ntohs(d), LF_MAC_ARGS(lag_id+15), ntohs(e), *(lag_id+23), ntohs(f));

  return s;
}
